//
// PROGRAM NAME:  POPUP.C.
//
// FUNCTIONAL DESCRIPTION.
//	This module comprises the C component of a popup demo program
//	that shows how to build a TSR (terminate and stay resident)
//	program to run under Embedded DOS.  It uses threads to run code
//	in the background while the prompt returns in the foreground.
//
//	This module is linked together with code from POPBASE.ASM to
//	form a popup TSR that runs stuff in the background and refreshes
//	its display when a key combination (configurable in POPBASE.ASM)
//	is detected on the keyboard.
//
// MODIFICATION HISTORY.
//	S. E. Jones	91/12/29.	#1.094, original.
//	S. E. Jones	93/09/10.	#1.106, moved to C 8.0.
//	S. E. Jones	95/08/17.	#6-XL v36, added _loadds to thread fn.
//
// NOTICE:  Copyright (C) 1991-1995 General Software, Inc.
//

#include <stdlib.h>
#include <string.h>
#include "../inc/ktypes.h"		// VOID, USHORT, etc.
#include "../inc/system.h"		// general STARLITE stuff.
#include "../inc/kernel.h"		// kernel API.

#define MAKETHREAD(f,sseg,pri) AllocateThread (f)

extern VOID InitializeHeap (UCHAR *BreakAddr, USHORT KbToUse);
extern UCHAR GetChar ();		// in POPBASE.ASM...
extern VOID SaveScreenMemory (UCHAR *Buffer);
extern VOID RestoreScreenMemory (UCHAR *Buffer);
extern VOID TaskBar (UCHAR *Msg, USHORT DisplayAttribute);
extern VOID ClrScr ();
extern VOID BlankScreen ();

//
// DOS-oriented functions implemented in LIB\DOSCALL.C.
// To access these routines, link with LIB\OBJ\DOSCALL.OBJ.
//

extern BOOLEAN IsStarlite ();
extern UCHAR GetSwitchChar ();
extern UCHAR SetSwitchChar (UCHAR Ch);
extern BOOLEAN GetBreakFlag ();
extern VOID SetBreakFlag (BOOLEAN Flag);
extern BOOLEAN GetVerifyFlag ();
extern VOID SetVerifyFlag (BOOLEAN Flag);
extern STATUS DosGetDate (USHORT *Day, USHORT *Date, USHORT *Month, USHORT *Year);
extern STATUS DosSetDate (USHORT Date, USHORT Month, USHORT Year);
extern STATUS DosGetTime (USHORT *Hours, USHORT *Minutes, USHORT *Seconds, USHORT *Hundredths);
extern STATUS DosSetTime (USHORT Hours, USHORT Minutes, USHORT Seconds);
extern UCHAR *DosGetDta ();
extern VOID DosSetDta (UCHAR *Dta);
extern STATUS DosExit (STATUS StatusCode);
extern STATUS DosDiskReset ();
extern STATUS DosCanonicalizePath (UCHAR *SrcPath, UCHAR *DestPath);
extern STATUS DosRename (UCHAR *SrcPath, UCHAR *DestPath);
extern STATUS DosGetDefaultDrive (USHORT *Drive);
extern STATUS DosSetDefaultDrive (USHORT Drive);
extern STATUS DosMkDir (UCHAR *Path);
extern STATUS DosRmDir (UCHAR *Path);
extern STATUS DosChDir (UCHAR *Path);
extern STATUS DosDelete (UCHAR *Path);
extern STATUS DosCreate (UCHAR *Path, USHORT *Handle);
extern STATUS DosOpen (UCHAR *Path, USHORT OpenMode, USHORT *Handle);
extern STATUS DosClose (USHORT Handle);
extern STATUS DosSetPos (USHORT Handle, ULONG Position, USHORT Method);
extern STATUS DosRead (USHORT Handle, UCHAR *Buffer, USHORT BufLen, USHORT *BytesRead);
extern STATUS DosWrite (USHORT Handle, UCHAR *Buffer, USHORT BufLen, USHORT *BytesWritten);
extern STATUS DosFindFirst (UCHAR *Path, USHORT Attributes);
extern STATUS DosFindNext ();
extern STATUS SetCookedMode (USHORT Handle);
extern STATUS DosAllocSeg (USHORT Paragraphs, UCHAR **BufPtr);
extern STATUS DosFreeSeg (UCHAR *BufPtr);

static USHORT Flags=0;
#define FLAGS_IN_POPUP	0x0001		// set when inside popup routine.

static UCHAR *SavedScreen=NULL;         // saved screen buffer.
static USHORT Counter=0;		// count fake things.

THREAD _loadds ThreadProc ()
{
    while (TRUE) {
	Counter++;
	if (!(Counter & 0x000f)) {
	    EnterCriticalSection ();	// giveup time slice if bottom bits are down.
	    LeaveCriticalSection ();
	}
    }
    DeallocateThread ();		// thread destructor; never returns.
} // ThreadProc

BOOLEAN SaveScreen ()
{
    if (SavedScreen != NULL) {
	return TRUE;
    }
    SavedScreen = malloc (80*25*2);
    if (SavedScreen == NULL) {
	return FALSE;
    }

    SaveScreenMemory (SavedScreen);
    return TRUE;
} // SaveScreen

VOID RestoreScreen ()
{
    if (SavedScreen == NULL) {
	return;
    }
    RestoreScreenMemory (SavedScreen);
    free (SavedScreen);
    SavedScreen = NULL;
} // RestoreScreen

VOID BuildScreenText (
    USHORT *Screen,
    UCHAR *Buffer,
    USHORT x,
    USHORT y,
    USHORT Attrib)
{
    USHORT *s;

    s = Screen + (x + y*80);

    while (*Buffer != 0) {
	*(s++) = (Attrib << 8) | (USHORT)*(Buffer++);
    }
} // BuildScreenText

VOID WriteScreen (
    UCHAR *Buffer,
    USHORT x,
    USHORT y)
{
    USHORT *Screen;
    ULONG Address;

    Address = 0xb8000000L;
    Screen = (USHORT *)Address;
    BuildScreenText (Screen, Buffer, x, y, 0x07);

    Address = 0xb0000000L;
    Screen = (USHORT *)Address;
    BuildScreenText (Screen, Buffer, x, y, 0x07);
} // WriteScreen

VOID DisplayInfo ()
{
    UCHAR LittleBuf [10];
    itoa (Counter, LittleBuf, 10);
    WriteScreen (LittleBuf, 38, 10);
} // DisplayInfo

VOID Popup ()
{
    BOOLEAN Done=FALSE;
    UCHAR ch;

    if (Flags & FLAGS_IN_POPUP) {
	return;
    }

    //
    // We just entered the popup routine.  Display a menu bar at the top
    // of the screen and then get on with the user's application.
    //

    if (SaveScreen ()) {
	Flags |= FLAGS_IN_POPUP;
	ClrScr ();			// zap entire screen now.
	TaskBar ("General Software TSR Popup:  (D)isplay or (E)xit", 0x71);

	while (!Done) {
	    ch = GetChar ();		// get char from BIOS.
	    switch (toupper (ch)) {
		case 'D':
		    DisplayInfo ();
		    break;
		case 'E':
		    Done = TRUE;
	    }
	}

	RestoreScreen ();
	Flags &= ~FLAGS_IN_POPUP;	// leaving the system.
    }
} // Popup

USHORT PopupInit (UCHAR *BreakAddr)	// FWA, free workspace area.
{
    USHORT KbToUse=8;			// nice round number.

    if (!IsStarlite ()) {
	printf ("This program requires Embedded DOS to run.\n");
	return 0;
    }

    //
    // Allocate a nice thread to run in the background.
    //

    MAKETHREAD (ThreadProc, 0, THREAD_PRIORITY_DEFAULT);

    //
    // Now tell the user we're installed.
    //

    printf ("General Software Popup Demo V1.0\n");
    printf ("Press Left-Shf and Right-Shf simultaneously to interrupt.\n");

    //
    // Return number of paragraphs to keep, this load.
    //

    InitializeHeap (BreakAddr, KbToUse);
    return (KbToUse*1024)/16;
} // PopupInit
