//***	KERNEL.H -- Embedded DOS Kernel API Definitions for C Language Programs.
//
//1.	Functional Description.
//	This include file contains prototypes for the Embedded DOS KERNEL API.
//
//2.	Modification History.
//	S. E. Jones	91/07/01.	Original for #1.085.
//	S. E. Jones	91/07/13.	#1.086, changed AllocateTimer ctx from PVOID to USHORT.
//	S. E. Jones	91/11/19.	#1.094, added long thread allocation.
//	S. E. Jones	92/01/03.	#1.094, added fast time slice pass.
//	S. E. Jones	92/01/04.	#1.095, added thread handles.
//	S. E. Jones	92/01/08.	#1.095, added fshelp/iohelp functions.
//	S. E. Jones	92/01/12.	#1.095, slstatus->status.
//	S. E. Jones	92/07/28.	#1.099, added QueryThreadHandle.
//	S. E. Jones	92/07/20.	#1.099, added object API.
//	S. E. Jones	93/01/08.	#1.101, added trace API.
//	S. E. Jones	93/03/14.	#1.101, added DOSDATA/PSP/DGROUP address functions.
//	S. E. Jones	93/04/01.	#1.102, moved debug stuff to chelper.
//	S. E. Jones	93/09/26.	#2.000, release.
//	S. E. Jones	95/01/24.	#2.2, release.
//	S. E. Jones	95/02/09.	#2.2, fixed prototype of point-to-object fn.
//	S. E. Jones	95/03/02.	#2.2.21, solved name mangling.
//
//3.	NOTICE: Copyright (C) 1991-1995 General Software, Inc.

#ifndef _KERNEL
#define _KERNEL

//
// This module depends on KTYPES.
//

#include "ktypes.h"		      // conditionally includes types.

#ifdef __BCPLUSPLUS__
#define SYSEXT extern "C"		// prevent "name mangling" in BCC.
#else
#define SYSEXT
#endif

//
// Thread Object Implementation.  The THREAD type (really just a macro) is
// defined here to add whatever is necessary to the definition of a function
// so that it will be called by the thread dispatcher with the correct setup.
// If DS needs to be loaded, for example, this is where it goes.
//

#define THREAD void			// for MS-C (change as necessary).

SYSEXT HANDLE AllocateThread (
    IN VOID (*Func)()
    );

SYSEXT HANDLE AllocateThreadLong (
    IN VOID (*Func)(),
    IN USHORT StackSegment,
    IN USHORT Priority
    );

SYSEXT HANDLE AllocateThreadLongEx (    // MVIMOD jmc 11/24/99
    IN VOID (*Func)(),
    IN VOID *StackTop,
    IN USHORT Priority
    );

SYSEXT STATUS DeallocateThread (void);
SYSEXT STATUS EnterCriticalSection (void);
SYSEXT STATUS LeaveCriticalSection (void);
SYSEXT STATUS PassTimeSlice (void);
SYSEXT STATUS AbortThread (HANDLE ThreadHandle);

SYSEXT STATUS SetThreadInformation (
    HANDLE ThreadHandle,
    USHORT ThreadInfoType,
    PVOID ThreadInformation);

SYSEXT STATUS QueryThreadInformation (
    HANDLE ThreadHandle,
    USHORT ThreadInfoType,
    PVOID ThreadInformation);

SYSEXT STATUS QueryThreadHandle (OUT PHANDLE Handle);  // returns handle to curr. thread.
SYSEXT STATUS PrioritizeThread (HANDLE ThreadHandle, USHORT NewPriority);

#define THREAD_PRIORITY_LOW		0	// lowest possible priority.
#define THREAD_PRIORITY_DEFAULT         16384	// standard timesliced priority.
#define THREAD_PRIORITY_HI		32767	// highest priority.
#define THREAD_PRIORITY_RESERVED	32768	// this and above are reserved.

#define THREAD_INFOTYPE_URUN		1	// user appendage for thread run.
#define THREAD_INFOTYPE_USTOP		2	// user appendage for thread block.
#define THREAD_INFOTYPE_UEXIT		3	// user appendage for thread exit.
#define THREAD_INFOTYPE_FLAGS		4	// capability flags.
#define THREAD_INFOTYPE_NAME		5	// thread's name (ASCIIZ string).
#define THREAD_INFOTYPE_P1		6	// user parameter #1.
#define THREAD_INFOTYPE_P2		7	// user parameter #2.

//
// Thread capability flags.
//

#define THREAD_CAPABILITY_386	0x0004		// 80386/80486 state preserved.
#define THREAD_CAPABILITY_X87	0x0008		// 80x87 state preserved.

//
// Named Object Implementation.
//

SYSEXT STATUS AllocateObject (
    IN UCHAR *AsciizName,
    IN PVOID UserValue,
    OUT PHANDLE ObjectHandle);

SYSEXT STATUS AccessObject (
    IN UCHAR *AsciizName,
    OUT PHANDLE ObjectHandle);

SYSEXT STATUS DeallocateObject (IN HANDLE ObjectHandle);
SYSEXT STATUS ReleaseObject (IN HANDLE ObjectHandle);
SYSEXT STATUS LockObject (IN HANDLE ObjectHandle);
SYSEXT STATUS UnlockObject (IN HANDLE ObjectHandle);

SYSEXT STATUS PointToObject (
    IN HANDLE ObjectHandle,
    OUT PVOID *UserValue);

//
// Event Object Implementation.
//

SYSEXT STATUS AllocateEvent (OUT PHANDLE EventHandle);
SYSEXT STATUS DeallocateEvent (IN HANDLE EventHandle);
SYSEXT STATUS SetEvent (IN HANDLE EventHandle);
SYSEXT STATUS SetEventIsr (IN HANDLE EventHandle);      // MVIMOD jmc 11/17/99
SYSEXT STATUS ClearEvent (IN HANDLE EventHandle);
SYSEXT STATUS PulseEvent (IN HANDLE EventHandle);
SYSEXT STATUS QueryEvent (IN HANDLE EventHandle, OUT USHORT *EventStatus);
SYSEXT STATUS WaitEvent (IN HANDLE EventHandle);

//
// Mutex Object Implementation.
//

SYSEXT STATUS AllocateMutex (OUT PHANDLE MutexHandle);
SYSEXT STATUS DeallocateMutex (IN HANDLE MutexHandle);
SYSEXT STATUS AcquireMutex (IN HANDLE MutexHandle);
SYSEXT STATUS ReleaseMutex (IN HANDLE MutexHandle);

//
// Spin Lock Object Implementation.  Note that the spin lock is
// defined as a 16-bit storage location, not a handle.	The user
// is required to maintain that location, and not transfer its
// contents to some other location.  The system uses the address
// of the spinlock, in addition to the contents, to maintain the lock.
//

SYSEXT STATUS AllocateSpinLock (USHORT *SpinLock);
SYSEXT STATUS DeallocateSpinLock (USHORT *SpinLock);
SYSEXT STATUS AcquireSpinLock (USHORT *SpinLock);
SYSEXT STATUS ReleaseSpinLock (USHORT *SpinLock);

//
// Timer Object.
//

SYSEXT STATUS AllocateTimer (
    OUT PHANDLE TimerHandle,
    IN USHORT Context,
    IN VOID (*ExpirationRtn)(USHORT)
    );

SYSEXT STATUS DeallocateTimer (IN HANDLE TimerHandle);
SYSEXT STATUS StartTimer (IN HANDLE TimerHandle, IN USHORT DeltaTime);
SYSEXT STATUS StopTimer (IN HANDLE TimerHandle);

//
// Kernel Memory Pool Object.
//

SYSEXT STATUS AllocatePool (IN USHORT BlockSize, OUT PVOID *BlockPointer);
SYSEXT STATUS DeallocatePool (IN PVOID BlockPointer);
SYSEXT STATUS KeepPool (IN PVOID BlockPointer);

//
// Addressability Functions.
//

SYSEXT STATUS GetFsHelpAddress (OUT STATUS (**FunctionPointer)());
SYSEXT STATUS GetIoHelpAddress (OUT STATUS (**FunctionPointer)());
SYSEXT STATUS GetDosDataDs (OUT PVOID *);
SYSEXT STATUS GetDosDataEs (OUT PVOID *);
SYSEXT STATUS GetDgroupDs (OUT PVOID *);
SYSEXT STATUS GetDgroupEs (OUT PVOID *);
SYSEXT STATUS GetPspDs (OUT PVOID *);
SYSEXT STATUS GetPspEs (OUT PVOID *);

//
// System Trace Functions.
//

SYSEXT STATUS ResetTrace (void);
SYSEXT STATUS DisableTrace (void);
SYSEXT STATUS EnableTrace (void);
SYSEXT STATUS PutTrace (USHORT TraceCode);

#endif // not defined, _KERNEL
