/*****************************************************************************
*   Copyright (c) 2011-2012 Prosoft Technology, Inc.
*
*   Title:        Communications Module Serial Port Definitions
*
*   Abstract:     This file contains definitions and declarations
*                 needed for serial port communication.
*
*   Environment:  Linux
*                 GCC C/C++ Compiler
*
*   Author:       HYU
*
*  08/20/2011 HYU - Migrated to ProLinx Linux platform.
*  08/26/2011 HYU - Added spRegisterCallback() for status monitoring.
*  02/08/2012 HYU - Added spGetDMACount().
*
*****************************************************************************/

#ifndef _PROSOFT_SPAPI_H
#define _PROSOFT_SPAPI_H

#include "plxdef.h"

/************************************
**    API FUNCTION RETURN CODES
*************************************/
#define SP_SUCCESS             0       // Function returned successfully
#define SP_ERR_BADPARAM        1       // A parameter is invalid
#define SP_ERR_REOPEN          2       // Device is already open
#define SP_ERR_NODEVICE        3       // Device is not present
#define SP_ERR_NOACCESS        4       // Invalid access
#define SP_ERR_TIMEOUT         5       // The function has timed out
#define SP_ERR_MSGTOOBIG       6       // The message is too large
#define SP_ERR_BADCONFIG       7       // The IO is not configured properly
#define SP_ERR_MEMALLOC        8       // Unable to allocate memory

/* Library function declarations */
#ifdef __cplusplus
#define SPAPIENTRY     extern "C" int
#else
#define SPAPIENTRY     int
#endif

typedef int SPAPIHANDLE;

#define SP_INVALID_HANDLE    (-1)

/***********************************
**  PHYSICAL LAYER DEFINITIONS
************************************/
#define     RS232         1
#define     RS485         2
#define     RS422         3

/***********************************
**  SERIAL PORT DEFINITIONS
************************************/
/* Com Ports */
#define     COM1            1
#define     COM2            2
#define     COM3            3
#define     COM4            4
#define     COM5            5

/* Baud Rates */
#define     BAUD_110        0
#define     BAUD_150        1
#define     BAUD_300        2
#define     BAUD_600        3
#define     BAUD_1200       4
#define     BAUD_2400       5
#define     BAUD_4800       6
#define     BAUD_9600       7
#define     BAUD_19200      8
#define     BAUD_28800      9
#define     BAUD_38400      10
#define     BAUD_57600      11
#define     BAUD_115200     12
#define     BAUD_230400     13

/* Parity */
#define     PARITY_NONE     0
#define     PARITY_ODD      1
#define     PARITY_EVEN     2
#define     PARITY_MARK     3
#define     PARITY_SPACE    4

/* Word Length */
#define     WORDLEN5        0
#define     WORDLEN6        1
#define     WORDLEN7        2
#define     WORDLEN8        3

/* Stopbits */
#define     STOPBITS1       0
#define     STOPBITS2       1

/* Handshaking Modes */
#define     HSHAKE_NONE     0x00
#define     HSHAKE_RTSCTS   0x01
#define     HSHAKE_DTRDSR   0x02
#define     HSHAKE_XONXOFF  0x04

typedef struct tagSPALTSETUP
{
    BYTE    baudrate;
    BYTE    parity;
    BYTE    wordlen;
    BYTE    stopbits;
    int     txquesize;      /* Transmit que size */
    int     rxquesize;      /* Receive que size  */
    BYTE    fifosize;       /* UART FIFO size */
} SP_ALT_SETUP;

/***********************************
**  OTHER DEFINITIONS
************************************/
/* States for RTS/CTS/DTR/DSR/DCD */
#define     ON              1
#define     OFF             0

/* Timeout values */
#define     TIMEOUT_ASAP    0
#define     TIMEOUT_FOREVER 0xFFFFFFFFL


/* Que Sizes */
#define     MINQSIZE        512     /* Minimum Que Size */
#define     DEFQSIZE		1024    /* Default Que Size */
#define     MAXQSIZE        16384   /* Maximum Que Size */

typedef enum
{
    SP_READ,
    SP_WRITE,
    SP_RTS_ON,
    SP_RTS_OFF,
    SP_CTS_ON,

}SP_COMM_STATUS;

/*************************************
**  API FUNCTION PROTOTYPES
*************************************/
typedef int (*PFUNC_SP_COMMCALLBACK)(int comport, int status, const unsigned char * buff, int buff_len);

SPAPIENTRY spRegisterCommCallback(PFUNC_SP_COMMCALLBACK spCallback);

SPAPIENTRY spOpen(int comport, BYTE baudrate, BYTE parity, BYTE wordlen, BYTE stopbits, SPAPIHANDLE * spHandle);
SPAPIENTRY spOpenAlt(int comport, SP_ALT_SETUP *altsetup, SPAPIHANDLE * spHandle);
SPAPIENTRY spClose(SPAPIHANDLE comport);

//SPAPIENTRY spConfig(SPAPIHANDLE comport, BYTE baudrate, BYTE parity, BYTE wordlen, BYTE stopbits); //not defined yet

SPAPIENTRY spGetPortPhysicalLayer(SPAPIHANDLE comport, int *setting);
SPAPIENTRY spSetPortPhysicalLayer(SPAPIHANDLE comport, int physicalLayer);

// This function allows to set the Automatic RTS option on the ATMEL processor
// state has to be ON or OFF
//SPAPIENTRY spSetAutomaticRTS(SPAPIHANDLE comport, int state);

SPAPIENTRY spSetRTS(SPAPIHANDLE comport, int state);
SPAPIENTRY spGetRTS(SPAPIHANDLE comport, int *state);
SPAPIENTRY spSetDTR(SPAPIHANDLE comport, int state);
SPAPIENTRY spGetDTR(SPAPIHANDLE comport, int *state);
SPAPIENTRY spGetCTS(SPAPIHANDLE comport, int *state);
//SPAPIENTRY spGetDSR(int comport, int *state);   //not defined yet
SPAPIENTRY spGetDCD(SPAPIHANDLE comport, int *state);   // defined for DNP
SPAPIENTRY spPutch(SPAPIHANDLE comport, BYTE ch, DWORD timeout);
SPAPIENTRY spPuts(SPAPIHANDLE comport, BYTE *str, BYTE term, int *len, DWORD timeout);
SPAPIENTRY spPutData(SPAPIHANDLE comport, BYTE *data, int *len, DWORD timeout);
SPAPIENTRY spGetch(SPAPIHANDLE comport, BYTE *ch, DWORD timeout);
SPAPIENTRY spGets(SPAPIHANDLE comport, BYTE *str, BYTE term, int *len, DWORD timeout); 
SPAPIENTRY spGetData(SPAPIHANDLE comport, BYTE *data, int *len, DWORD timeout);
SPAPIENTRY spGetCountUnsent(SPAPIHANDLE comport, int *count);
SPAPIENTRY spGetCountUnread(SPAPIHANDLE comport, int *count);
SPAPIENTRY spSetHandshaking(SPAPIHANDLE comport, int shake);
SPAPIENTRY spPurgeDataUnsent(SPAPIHANDLE comport);
SPAPIENTRY spPurgeDataUnread(SPAPIHANDLE comport);
SPAPIENTRY spGetDMACount(SPAPIHANDLE comport, unsigned short int *count);

#endif //#ifndef _PROSOFT_SPAPI_H

