////////////////////////////////////////////////////////////////////////////////
// MODIFIED:
// 12/02/2008 RAR - Altered code so that CIP backplane and application data is
//                  all local to the functions.  This method does not require
//                  the backplane interface functions and application data in
//                  order for the service to function.  But, it now requires
//                  the application to initialize the data structure before
//                  calling the initDiscoveryService() function.
//
// 05/23/2011 HYU - Changed to send the response with broadcast mode so that PDS
//                  can find devices even though they are not on the same network.
//
// 05/30/2011 HYU - Modified ChangeIPAddrsRequest() to add the call to addBroadcastRoute()
//                  after IP changed because it will be lost after the IP address is changed.
//                  This ensures that the broadcast UDP message is sent successfully.
//
// 05/31/2011 HYU - The new netmask will not be saved if calling setNetworkMask failed.
// 06/12/2011 HYU - Modified SaveIpConfig() to fix the corrupt config file issue.
//                  The root cause is that Linux file system cannot sync file changes
//                  to SD card when the size of data changed is less than the cache size.
//                  The workaround is to create a new file and then rename it, finally
//                  call sync.
// 06/14/2011 HYU - Modified SaveIpConfig() to use sync() instead of system call "sync".
//                  Modified processDiscoveryServiceMsg() to send the response only when
//                  the request is handled successfully.
// 06/17/2011 HYU - Added the RebootModule command. Borrowed the code of shutdownSystem()
//                  from web service.
// 08/20/2011 HYU - Migrated to ProLinx Linux platform.
// 11/01/2011 HYU - Modified SaveIpConfig to store the ip configuration into the wattcp.cfg file.
// 10/17/2012 HYU - Used saveWattcpNetworkDataEx() to save the ip configuration.
// 10/23/2012 HYU - Removed the calls to addBroadcastRoute() because the socket will be bound to eth0 interface
//                  by setting the option SO_BINDTODEVICE. Also the broadcast route will affect the wireless broadcast.
// 11/29/2012 HYU - Implemented DSPortMonitor to monitor multiple Ethernet ports.
// 12/16/2015 JOA - Modified processDiscoveryServiceMsg function to fix  TEG 16458:
//                  Discovery Service Firmware UDP Broadcast Storm Vulnerability.
//
////////////////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <signal.h>
#include <fcntl.h>
#include <pthread.h>
#include <sys/ioctl.h>
#include <net/if.h>
#include <ifaddrs.h>

#include "ocxbpapi.h"

#include "discoveryservice.h"
#include "nethelper.h"
#include "moduleinfo.h"
#include "wattcpedit.h"



#define DISCOVERY_SERVICE_PORT 50000
#define MAX_BUF_LEN              200   //Maximum local buffer length
#define MAX_MESSAGE             1400   //Maximum message length
#define MAX_FIELDS                20   //Maximum number of fields in a message

#define PDS_POLLING_TIMEOUT   5000 //in ms

// These are the regular and temporal interfaces used by this service
#define PDS_IF0_NAME       "eth0"
#define PDS_TEMP_IF0_NAME  "eth0:9"
#define PDS_IF1_NAME       "eth1"
#define PDS_TEMP_IF1_NAME  "eth1:9"

#define PDS_MAX_PORT_NUM    2


typedef struct DSPortMonitor
{
    int dsSocket;
    char * ifname;
    char * temp_ifname;
} DSPortMonitor;


// used to control the number of Ethernet ports needed to config
static int m_numPortConfig = 0;

static int m_LastSocket = -1;
static fd_set m_rfds;

static DSPortMonitor m_dsPortMonitor[PDS_MAX_PORT_NUM];
static MODULE_INFO   mi_data;

static pthread_t dsThread;



static int DSPortMonitor_Init(DSPortMonitor * ds, int blocking);
static int DSPortMonitor_Process(DSPortMonitor * ds);

static int startDiscoveryServiceThread(void);




/*****************************************************************************
 * Purpose: Function that orders the linux system to shutdown
 *
 * Params: Nothing
 *
 * Returns: 0 if OK
 *
 * Comments:
 *
 *
******************************************************************************/
int shutdownSystem(void)
{
    pid_t pid;

    signal(SIGCLD, SIG_IGN);  // now I don't have to wait()

    switch(pid=fork())
    {
    case -1: // Something went wrong
        perror("fork");
        return -1;
    case 0:  // this is the child process
        //sleep(2); // Wait so at least this page can be returned
        // In order to be able to do a shutdown the process has to be root
        // because the web server doesn't run as root, the suid bit has to be added
        // to the plx80upgrade
        setuid(geteuid());
        // Kill init process
        kill(1,SIGTERM);
        exit(0);
    default: // This is the parent process
        return 0;
    }
    return(1);
}


typedef struct _DeviceIpInfo
{
    char id[16];
    char ip[16];
    char netMask[16];
    char gateway[16];
}DeviceIpInfo;


int FindIpConfig(DeviceIpInfo ipInfo[], int count, char* id)
{
    int i = 0;
    for(; i < count; i++ )
    {
        if (strcmp(ipInfo[i].id, id) == 0)
        {
            return i;
        }
    }
    
    return -1;
}


void SaveIpConfig(DeviceIpInfo ipInfo[], int count)
{
    IpConfigEx ipConfig[PDS_MAX_PORT_NUM];
    int numConfig = 0;
    int index = 0;
    
    index = FindIpConfig(ipInfo, count, PDS_IF0_NAME);
    if ( index != -1)
    {
        numConfig++;
        ipConfig[0].type = IP_CFG_PORT1;
        ipConfig[0].ip = ipInfo[index].ip;
        ipConfig[0].netmask = ipInfo[index].netMask;
        ipConfig[0].gateway = ipInfo[index].gateway;
        
#ifdef DEBUG
        printf("E1: IP changed: %s %s\n", ipInfo[index].ip, ipInfo[index].netMask);
#endif
    }
    
    index = FindIpConfig(ipInfo, count, PDS_IF1_NAME);
    if ( index != -1)
    {
        numConfig++;
        ipConfig[1].type = IP_CFG_PORT2;
        ipConfig[1].ip = ipInfo[index].ip;
        ipConfig[1].netmask = ipInfo[index].netMask;
        ipConfig[1].gateway = ipInfo[index].gateway;
        
#ifdef DEBUG
        printf("E2: IP changed: %s %s\n", ipInfo[index].ip, ipInfo[index].netMask);
#endif
    }
    
    saveWattcpNetworkDataEx(ipConfig, numConfig);
    
}


/******************************************************************************
 * Purpose: This function initializes the discovery service
 *
 * Params: Number of port supported
 *
 * Returns:  0 if OK
 *          -1 on Error
 *
 * Comments:
 *
 *****************************************************************************/
int initDiscoveryService(int numPort)
{
    GetModuleInfo(&mi_data);

    FD_ZERO(&m_rfds);

    m_numPortConfig = 0;

    if (numPort > 0 && numPort <= PDS_MAX_PORT_NUM)
    {
        m_dsPortMonitor[0].dsSocket = -1;
        m_dsPortMonitor[0].ifname = PDS_IF0_NAME;
        m_dsPortMonitor[0].temp_ifname = PDS_TEMP_IF0_NAME;
        if (DSPortMonitor_Init(&m_dsPortMonitor[0], 1) == 0)
        {
            m_numPortConfig++;
            FD_SET(m_dsPortMonitor[0].dsSocket, &m_rfds);
            if (m_LastSocket < m_dsPortMonitor[0].dsSocket)
            {
                m_LastSocket = m_dsPortMonitor[0].dsSocket;
            }
        }
    }

    if (numPort > 1 && numPort <= PDS_MAX_PORT_NUM)
    {
        m_dsPortMonitor[1].dsSocket = -1;
        m_dsPortMonitor[1].ifname = PDS_IF1_NAME;
        m_dsPortMonitor[1].temp_ifname = PDS_TEMP_IF1_NAME;
        if (DSPortMonitor_Init(&m_dsPortMonitor[1], 1) == 0)
        {
            m_numPortConfig++;
            FD_SET(m_dsPortMonitor[1].dsSocket, &m_rfds);
            if (m_LastSocket < m_dsPortMonitor[1].dsSocket)
            {
                m_LastSocket = m_dsPortMonitor[1].dsSocket;
            }
        }
    }

    return startDiscoveryServiceThread();
}


/******************************************************************************
 * Purpose: This function initializes the discovery service specifying if
 *          it will block or not.
 *
 * Params: None
 *
 * Returns:  0 if OK
 *          -1 on Error
 *
 * Comments:
 *
 *****************************************************************************/
static int DSPortMonitor_Init(DSPortMonitor * ds, int blocking)
{
    struct sockaddr_in my_addr;   // my address information
    int broadcast = 1;
    
    
    // Create the Socket
    if ((ds->dsSocket = socket(PF_INET, SOCK_DGRAM, 0)) == -1)
    {
        perror("socket in discoveryService.c");
        return -1;
    }
    
    if( setsockopt( ds->dsSocket, SOL_SOCKET, SO_BINDTODEVICE, ds->ifname, strlen(ds->ifname) ) == -1 )
    {
        perror("setsockopt SO_BINDTODEVICE");
        return -1;
    }
    
    // Configure the socket so it can send broadcast messages
    if (setsockopt(ds->dsSocket, SOL_SOCKET, SO_BROADCAST, &broadcast, sizeof(broadcast)) == -1)
    {
        perror("setsockopt in discoveryService.c");
        return -1;
    }
    
    // Set to non blocking
    if (blocking==0)
    {
        if (fcntl(ds->dsSocket, F_SETFL, O_NONBLOCK) == -1)
        {
            perror("fcntl in discoveryService.c");
            return -1;
        }
    }
    
    // Set the address to bind to
    my_addr.sin_family = AF_INET; // host byte order
    my_addr.sin_port = htons(DISCOVERY_SERVICE_PORT); // short, network byte order
    // This would be the standard for a service that will run in every port
    my_addr.sin_addr.s_addr = INADDR_ANY; // automatically fill with my IP
    memset(&(my_addr.sin_zero), '\0', 8); // zero the rest of the struct
    
    // Proceed to BIND
    if (bind(ds->dsSocket, (struct sockaddr *)&my_addr, sizeof(struct sockaddr)) == -1)
    {
        perror("setsockopt in discoveryService.c");
        return -1;
    }
    
    return 0;
}


void sendDiscoverResponse(DSPortMonitor * ds, struct sockaddr_in originAddr)
{
    char send_buff[MAX_MESSAGE];   //data area used to hold message
    int  send_buff_len = 0;        //number of characters to print
    struct in_addr eth0_ip;        // eth0 IP address
    struct in_addr eth0_netmask;   // eth0 Network Mask
    struct in_addr eth0_9_ip;      // eth0:9 temporary IP address
    struct in_addr eth0_9_netmask; // eth0:9 Network Mask
    struct sockaddr_in destAddr;  // response destination address
    int numbytes;
    unsigned char macAddr[6];
    
    // Get all the data required to answer the DiscoverRQ Message
    getIPAddress(ds->ifname, &eth0_ip);
    getNetworkMask(ds->ifname, &eth0_netmask);
    getIPAddress(ds->temp_ifname, &eth0_9_ip);
    getNetworkMask(ds->temp_ifname, &eth0_9_netmask);
    getMACAddress(ds->ifname, macAddr);
    
    send_buff_len = snprintf(send_buff, MAX_MESSAGE-1,
        "DiscoverRS\n"    // Response Code {0}
        "%s\n"			// Product Type  {1}
        "%s\n"			// Product Name  {2}
        "%08lX\n"     		// Serial Number {3}
        "%02X:%02X:%02X:%02X:%02X:%02X\n" // MAC Address {4}
        "%s\n"			// Module Name   {5}
        "%s\n"		// Product Name Code {6}
        "%s\n"		// Product Revision  {7}
        "%s\n"		// Software Revision Level {8}
        "%s\n"		// OS Revision Level  {9}
        "%u.%u.%u.%u\n"	// Main IP Address    {10}
        "%u.%u.%u.%u\n"	// Main IP NetMask    {11}
        "%u.%u.%u.%u\n"	// Temporary IP Address {12}
        "%u.%u.%u.%u\n",  // Temporary IP Netmask {13}
        mi_data.ProductType,   			//Product Type
        mi_data.ProductName,				//Product Name
        mi_data.SerialNo,   		//Serial Number
        macAddr[5],macAddr[4],macAddr[3],macAddr[2],macAddr[1],macAddr[0], // MAC Address
        mi_data.ModuleName,		// Module Name
        mi_data.ProductName,
        mi_data.ProductRev,
        mi_data.OpRev,
        mi_data.OpRun,
        (eth0_ip.s_addr & 0xff), ((eth0_ip.s_addr >> 8) & 0xff), ((eth0_ip.s_addr >> 16) & 0xff), ((eth0_ip.s_addr >> 24) & 0xff),
        (eth0_netmask.s_addr & 0xff), ((eth0_netmask.s_addr >> 8) & 0xff), ((eth0_netmask.s_addr >> 16) & 0xff), ((eth0_netmask.s_addr >> 24) & 0xff),
        (eth0_9_ip.s_addr & 0xff),((eth0_9_ip.s_addr >> 8) & 0xff), ((eth0_9_ip.s_addr >> 16) & 0xff), ((eth0_9_ip.s_addr >> 24) & 0xff),
        (eth0_9_netmask.s_addr & 0xff),((eth0_9_netmask.s_addr >> 8) & 0xff), ((eth0_9_netmask.s_addr >> 16) & 0xff), ((eth0_9_netmask.s_addr >> 24) & 0xff));
    
#ifdef DEBUG
    printf("Sending Response: \n %s \n",send_buff);
#endif // DEBUG
    
    destAddr.sin_family = AF_INET; // host byte order
    destAddr.sin_port = originAddr.sin_port; // respond to the same port used in the request
    
    // Set Broadcast Address
    destAddr.sin_addr.s_addr = INADDR_BROADCAST;
    memset(&(destAddr.sin_zero), '\0', 8); // zero the rest of the struct
    
    // send transaction response
    if ((numbytes=sendto(ds->dsSocket, send_buff, send_buff_len, 0,
        (struct sockaddr *)&destAddr, sizeof(struct sockaddr))) == -1)
    {
        perror("sendto in discoveryService.c");
        return;
    }
}


/******************************************************************************
 * Purpose: This function sends a response to the discovery tool
 *
 * Params: originAddr: The remote address
 *
 * Returns:  None
 *
 * Comments: Uses the new protocol to provide more information in order to support the changes to permanent IP.
 *
 *
 *****************************************************************************/
void sendDiscoverResponse2(DSPortMonitor * ds, struct sockaddr_in originAddr)
{
    char send_buff[MAX_MESSAGE];   //data area used to hold message
    int  send_buff_len = 0;        //number of characters to print
    struct in_addr ip;        // eth0 IP address
    struct in_addr netmask;   // eth0 Network Mask
    struct in_addr loAddr;    // localhost address
    int numbytes;
    unsigned char macAddr[6];
    struct ifaddrs *ifaddr, *ifa;
    int family;
    struct sockaddr_in destAddr;  // response destination address
    
    destAddr.sin_family = AF_INET; // host byte order
    destAddr.sin_port = originAddr.sin_port; // respond to the same port used in the request
    
    // Set Broadcast Address
    destAddr.sin_addr.s_addr = INADDR_BROADCAST;
    memset(&(destAddr.sin_zero), '\0', 8); // zero the rest of the struct
    
    // Get all the data required to answer the DiscoverRQ Message
    
    send_buff_len = snprintf(send_buff, MAX_MESSAGE-1,
        "DiscoverRS2\n" // Response Code {0}
        "%s\n"			// Product Type  {1}
        "%s\n"			// Product Name  {2}
        "%08lX\n"     	// Serial Number {3}
        "%s\n"			// Module Name   {4}
        "%s\n"		// Product Name Code {5}
        "%s\n"		// Product Revision  {6}
        "%s\n"		// Software Revision Level {7}
        "%s\n",		// OS Revision Level  {8}
        mi_data.ProductType,   		//Product Type
        mi_data.ProductName,		//Product Name
        mi_data.SerialNo,   		//Serial Number
        mi_data.ModuleName,		// Module Name
        mi_data.ProductCode,
        mi_data.ProductRev,
        mi_data.OpRev,
        mi_data.OpRun);
    
    if (getifaddrs(&ifaddr) == -1)
    {
        perror("getifaddrs");
        exit(EXIT_FAILURE);
    }
    
    inet_aton("127.0.0.1",&loAddr);
    
    /* Walk through linked list, maintaining head pointer so we can free list later */
    for (ifa = ifaddr; ifa != NULL; ifa = ifa->ifa_next)
    {
        if (ifa->ifa_addr == NULL)
        {
            continue;
        }

        family = ifa->ifa_addr->sa_family;
        if (family == AF_INET)
        {
            if (getMACAddress(ifa->ifa_name, macAddr) != 0)
            {
                // ignore the entry
                continue;
            }
            
            ip = ((struct sockaddr_in *)ifa->ifa_addr)->sin_addr;
            
            if (loAddr.s_addr == ip.s_addr)
            {
                // ignore the localhost entry
                continue;
            }
            
            if (m_numPortConfig == 1 && strstr(ifa->ifa_name, PDS_IF0_NAME) == NULL)
            {
                continue;
            }
            
            netmask = ((struct sockaddr_in *)ifa->ifa_netmask)->sin_addr;
            
            send_buff_len += snprintf(send_buff + send_buff_len, MAX_MESSAGE-1 - send_buff_len,
                "%s\n" // Ifc name
                "%02X:%02X:%02X:%02X:%02X:%02X\n" // MAC Address
                "%u.%u.%u.%u\n"	// IP Address
                "%u.%u.%u.%u\n",	// IP NetMask
                ifa->ifa_name,
                macAddr[5],macAddr[4],macAddr[3],macAddr[2],macAddr[1],macAddr[0], // MAC Address
                (ip.s_addr & 0xff), ((ip.s_addr >> 8) & 0xff), ((ip.s_addr >> 16) & 0xff), ((ip.s_addr >> 24) & 0xff),
                (netmask.s_addr & 0xff), ((netmask.s_addr >> 8) & 0xff), ((netmask.s_addr >> 16) & 0xff), ((netmask.s_addr >> 24) & 0xff));
        }
    }
    
    freeifaddrs(ifaddr);
    
#ifdef DEBUG
    printf("Sending Response: \n %s \n",send_buff);
#endif // DEBUG
    
    // Try to send the transaction response via broadcast.
    if ((numbytes = sendto(ds->dsSocket, send_buff, send_buff_len, 0,
        (struct sockaddr *)&destAddr, sizeof(struct sockaddr))) == -1)
    {
        perror("failed to send the response via broadcast");
    }
}


/******************************************************************************
 * Purpose: This function process a ChangeTempIP Request message
 *
 * Params: char *fields[]: Fields received in the message
 * 		   int fieldcount: Number of fields available
 *
 * Returns:  None
 *
 * Comments: The content of the fields for this command should be as follows:
 *
 *	 		fields[0] = ChangeTempIPRQ
 * 			fields[1] = Serial Number of the device that has to change the Temp IP
 * 			fields[2] = MAC address of the device that has to change the Temp IP
 * 			fields[3] = new Temporary IP Address
 * 			fields[4] = new Temporary Network Mask
 *
 *				In order to change the IP address, both serial number and
 * 				MAC address must match
 *
 *
 *****************************************************************************/
int ChangeTempIPRequest(DSPortMonitor * ds, char *fields[], int fieldCount)
{
    char myMACAddressString[20];
    unsigned char macAddr[6];
    unsigned long msgSerial;
    struct in_addr ipAddr;
    
    // Check that all required fields are received
    if (fieldCount < 5)
    {
#ifdef DEBUG
        printf("Missing Parameters: %d\n",fieldCount);
#endif
        return -1;
    }
    
    // Get the  MAC Address
    getMACAddress(PDS_IF0_NAME, macAddr);
    snprintf(myMACAddressString, 20,"%02X:%02X:%02X:%02X:%02X:%02X",macAddr[5],
        macAddr[4],
        macAddr[3],
        macAddr[2],
        macAddr[1],
        macAddr[0]);
    //msgSerial = strtoul(fields[1],NULL,10);
    sscanf(fields[1],"%lX",&msgSerial);
    
    
    // Serial Number & MAC address have to match
    if (msgSerial != mi_data.SerialNo)
    {
#ifdef DEBUG
        printf("Received Serial Number doesn't match: %s %lu\n",fields[1], mi_data.SerialNo);
#endif
        return -1;
    }
    if (strcmp(myMACAddressString, fields[2]) != 0)
    {
#ifdef DEBUG
        printf("MAC Address doesn't match: %s %s\n",fields[2], myMACAddressString);
#endif
        return -1;
    }
    
    // If reached this point everything has matched so proceed to change the Temporary IP
    if (inet_aton(fields[3], &ipAddr)==0)
    {
#ifdef DEBUG
        printf("Invalid Temporary IP Received: %s\n",fields[3]);
#endif
        return -1;
    }
    
    //setIPAddress(ds->temp_ifname, &ipAddr);
    setIPAddress(PDS_TEMP_IF0_NAME, &ipAddr);
    
    if (inet_aton(fields[4], &ipAddr)==0)
    {
#ifdef DEBUG
        printf("Invalid Network Mask IP Received: %s\n",fields[4]);
#endif
        return -1;
    }
    
    //setNetworkMask(ds->temp_ifname, &ipAddr);
    setNetworkMask(PDS_TEMP_IF0_NAME, &ipAddr);
    
    return 0;
}


/******************************************************************************
 * Purpose: This function process a ChangeTempIP Request message
 *
 * Params: char *fields[]: Fields received in the message
 * 		   int fieldcount: Number of fields available
 *
 * Returns:  None
 *
 * Comments: The content of the fields for this command should be as follows:
 *
 *	 		fields[0] = RemoveTempIPRQ
 * 			fields[1] = Serial Number of the device that has to change the Temp IP
 * 			fields[2] = MAC address of the device that has to change the Temp IP
 *
 *				In order to change the IP address, both serial number and
 * 			MAC address must match
 *
 *
 *****************************************************************************/
int RemoveTempIPRequest(DSPortMonitor * ds, char *fields[], int fieldCount)
{
    char myMACAddressString[20];
    unsigned char macAddr[6];
    unsigned long msgSerial;
    
    // Check that all required fields are received
    if (fieldCount < 3)
    {
#ifdef DEBUG
        printf("Missing Parameters: %d\n",fieldCount);
#endif
        return -1;
    }
    
    // Get the MAC Address
    getMACAddress(PDS_IF0_NAME, macAddr);
    snprintf(myMACAddressString, 20,"%02X:%02X:%02X:%02X:%02X:%02X",macAddr[5],
        macAddr[4],
        macAddr[3],
        macAddr[2],
        macAddr[1],
        macAddr[0]);
    //msgSerial = strtoul(fields[1],NULL,10);
    sscanf(fields[1],"%lX",&msgSerial);
    
    // Serial Number & MAC address have to match
    if (msgSerial != mi_data.SerialNo)
    {
#ifdef DEBUG
        printf("Received Serial Number doesn't match: %s %lu\n",fields[3],mi_data.SerialNo);
#endif
        return -1;
    }
    if (strcmp(myMACAddressString,fields[2]) != 0)
    {
#ifdef DEBUG
        printf("MAC Address doesn't match: %s %s\n",fields[3], myMACAddressString);
#endif
        return -1;
    }
    
#ifdef DEBUG
    printf("Remove temp IP!\n");
#endif
    
    //ifdown(ds->temp_ifname);
    ifdown(PDS_TEMP_IF0_NAME);
    
    return 0;
}


/******************************************************************************
 * Purpose: This function process a ChangePermanentIP Request message
 *
 * Params: char *fields[]: Fields received in the message
 * 		   int fieldcount: Number of fields available
 *
 * Returns:  None
 *
 * Comments: The content of the fields for this command should be as follows:
 *
 *	 		fields[0] = ChangeIPAddrsRQ
 * 			fields[1] = Serial Number of the device that has to change the IP
 * 			fields[2] = Ifc name (eth0)
 * 			fields[3] = MAC address of (eth0)
 * 			fields[4] = new IP Address
 * 			fields[5] = new Network Mask
 * 			fields[6] = Ifc name (eth1)
 * 			fields[7] = MAC address of (eth1)
 * 			fields[8] = new IP Address
 * 			fields[9] = new Network Mask
 *
 *				In order to change the IP address, both serial number and
 * 				MAC address must match
 *
 *
 *****************************************************************************/
int ChangeIPAddrsRequest(DSPortMonitor * ds, char *fields[], int fieldCount)
{
    char myMACAddressString[20];
    unsigned char macAddr[6];
    unsigned long msgSerial;
    struct in_addr ipAddr, defaultGateway;
    int startIndex = 2;
    DeviceIpInfo ipInfo[(MAX_FIELDS - 2)/4];
    int offset = 0;
    
    memset(ipInfo, 0, sizeof(ipInfo));
    
    // Check that all required fields are received
    if (fieldCount < 6)
    {
#ifdef DEBUG
        printf("Missing Parameters: %d\n",fieldCount);
#endif
        return -1;
    }
    
    sscanf(fields[1],"%lX",&msgSerial);
    
    // Serial Number & MAC address have to match
    if (msgSerial != mi_data.SerialNo)
    {
#ifdef DEBUG
        printf("Received Serial Number doesn't match: %s %lu\n",fields[1], mi_data.SerialNo);
#endif
        return -1;
    }
    
    for (;(startIndex + 4) <= fieldCount; startIndex += 4)
    {
        // Get the  MAC Address
        getMACAddress(fields[startIndex], macAddr);
        snprintf(myMACAddressString, 20, "%02X:%02X:%02X:%02X:%02X:%02X", macAddr[5],
            macAddr[4],
            macAddr[3],
            macAddr[2],
            macAddr[1],
            macAddr[0]);
        
        if (strcmp(myMACAddressString, fields[startIndex + 1]) != 0)
        {
#ifdef DEBUG
            printf("MAC Address doesn't match: %s %s\n",fields[startIndex + 1], myMACAddressString);
#endif
            continue;
        }
        
        // If reached this point everything has matched so proceed to change the IP
        if (inet_aton(fields[startIndex + 2], &ipAddr)==0)
        {
#ifdef DEBUG
            printf("Invalid IP Received: %s\n",fields[startIndex + 2]);
#endif
            continue;
        }
        
        defaultGateway = ipAddr;
        
        setIPAddress(fields[startIndex], &ipAddr);
        
        if (inet_aton(fields[startIndex + 3], &ipAddr)==0)
        {
#ifdef DEBUG
            printf("Invalid Network Mask IP Received: %s\n",fields[startIndex + 3]);
#endif
            continue;
        }
        
        defaultGateway.s_addr &= ipAddr.s_addr;
        defaultGateway.s_addr |= 0x01000000; // generate an IP address for default gateway
        
        if (setNetworkMask(fields[startIndex], &ipAddr) != 0)
        {
            ipInfo[offset].netMask[0] = 0;
#ifdef DEBUG
            printf("Failed to change Network Mask IP: %s\n",fields[startIndex + 3]);
#endif
        }
        else
        {
            strncpy(ipInfo[offset].netMask, fields[startIndex + 3], 15);
        }
        
        strncpy(ipInfo[offset].id, fields[startIndex], 15);
        strncpy(ipInfo[offset].ip, fields[startIndex + 2], 15);
        
        strncpy(ipInfo[offset].gateway, inet_ntoa(defaultGateway), 15);
        
        offset++;
    }
    
    SaveIpConfig(ipInfo, offset);
    
    return 0;
}


/******************************************************************************
 * Purpose: This function process a RebootModule Request message
 *
 * Params: char *fields[]: Fields received in the message
 * 		   int fieldcount: Number of fields available
 *
 * Returns:  None
 *
 * Comments: The content of the fields for this command should be as follows:
 *
 *	 		fields[0] = RebootModuleRQ
 * 			fields[1] = Serial Number of the device
 * 			fields[2] = MAC address of the device
 *
 *				In order to reboot the module, both serial number and
 * 			MAC address must match
 *
 *
 *****************************************************************************/
int RebootModuleRequest(DSPortMonitor * ds, char *fields[], int fieldCount)
{
    char myMACAddressString[20];
    unsigned char macAddr[6];
    unsigned long msgSerial;
    
    // Check that all required fields are received
    if (fieldCount < 3)
    {
#ifdef DEBUG
        printf("Missing Parameters: %d\n",fieldCount);
#endif
        return -1;
    }
    
    // Get the MAC Address
    getMACAddress(PDS_IF0_NAME, macAddr);
    snprintf(myMACAddressString, 20,"%02X:%02X:%02X:%02X:%02X:%02X",macAddr[5],
        macAddr[4],
        macAddr[3],
        macAddr[2],
        macAddr[1],
        macAddr[0]);
    //msgSerial = strtoul(fields[1],NULL,10);
    sscanf(fields[1],"%lX",&msgSerial);
    
    // Serial Number & MAC address have to match
    if (msgSerial != mi_data.SerialNo)
    {
#ifdef DEBUG
        printf("Received Serial Number doesn't match: %s %lu\n",fields[3],mi_data.SerialNo);
#endif
        return -1;
    }
    if (strcmp(myMACAddressString,fields[2]) != 0)
    {
#ifdef DEBUG
        printf("MAC Address doesn't match: %s %s\n",fields[3], myMACAddressString);
#endif
        return -1;
    }
    
#ifdef DEBUG
    printf("Reboot module!\n");
#endif
    
    shutdownSystem();
    
    return 0;
}


/******************************************************************************
 * Purpose: This function process a received discovery service message
 *
 * Params: char *msg:  Message Data
 *         int msglen: Message Length
 *         struct sockaddr_in originAddr: Sender Address
 *
 * Returns:  None
 *
 * Comments: The Discovery Service Protocol is very simple. Is a list of '\n'
 *           separated strings being the first string the command function. For
 * 	         every request there is a response that follows the same format.
 *
 * 			 Commands:
 * 				DiscoverRQ : Browse network. This command is sent by a Browser looking for
 *                           devices on the network.
 * 				DiscoverRS : It contains the product data following this format:
 * 				   Value                         Data Example
 * 					[Product Type]<LF>            ILX56<LF>
 * 				 	[Product Name]<LF>            MVI56-MCM Rev. 2<LF>
 *                [Serial Number]<LF>           12345608<LF>
 *                [MAC Address}<LF>             00:11:22:33:44:55<LF>
 *      	         [User Assigned Name]<LF>      Test Module Name<LF>
 * 					[Product Code Name]<LF>       MCM5<LF>
 *              	[Product revision]<LF>        2.01<LF>
 *                [Product Op Code]<LF>         0908<LF>
 *                [Product Run Number]<LF>      0401<LF>
 * 					[Main IP Address]<LF>         192.168.9.253<LF>
 * 					[Main network mask]<LF>       255.255.255.0<LF>
 * 					[Temporary IP Address]<LF>    192.168.8.253<LF>
 * 					[Temporary network mask]<LF>  255.255.255.0<LF>
 *
 *
 *
 *****************************************************************************/
void processDiscoveryServiceMsg(DSPortMonitor * ds, char *msg, int msglen, struct sockaddr_in originAddr)
{
    char * fields[MAX_FIELDS];    // This array will hold all the fields
    char * currentField = msg;
    int    fieldCount = 0;
    int    retCmd = -1;           // 12/16/2015 JOA - Modified.
    
    // Look through all the fields
    while (fieldCount < MAX_FIELDS)
    {
        fields[fieldCount] = currentField;
        fieldCount++;
        currentField = strstr(currentField,"\n");
        if (currentField != NULL)
        {
            *currentField = 0;
            currentField++;
        }
        else
        {
            break;
        }
    }
    
#ifdef DEBUG
    printf("\nReceived Fields:\n");
    int i;
    for (i=0; i < fieldCount; i++)
    {
        printf("\t%s\n", fields[i]);
    }
#endif
    
    if (strcmp(fields[0], "DiscoverRQ")==0)
    {
        retCmd = 0;     // 12/16/2015 JOA - Added.
    }
    else if (strcmp(fields[0], "ChangeTempIPRQ")==0)
    {
        retCmd = ChangeTempIPRequest(ds, fields, fieldCount);
    }
    else if (strcmp(fields[0], "RemoveTempIPRQ")==0)
    {
        retCmd = RemoveTempIPRequest(ds, fields, fieldCount);
    }
    else if (strcmp(fields[0], "ChangeIPAddrsRQ")==0)
    {
        retCmd = ChangeIPAddrsRequest(ds, fields, fieldCount);
    }
    else if (strcmp(fields[0], "RebootModuleRQ")==0)
    {
        retCmd = RebootModuleRequest(ds, fields, fieldCount);
    }
    
    // Answer with a discover response
    if (retCmd == 0)
    {
#if (PDS_MAX_PORT_NUM == 2)
        sendDiscoverResponse2(ds, originAddr);
#else
        sendDiscoverResponse(ds, originAddr);
#endif
    }
}


static int DSPortMonitor_Process(DSPortMonitor * ds)
{
    socklen_t addr_len;
    struct sockaddr_in originAddr;  // connector address information
    int numbytes;
    char buf[MAX_BUF_LEN];

    // Store the addr field length
    addr_len = sizeof(struct sockaddr);
    
    // Receive data from the socket
    if ((numbytes=recvfrom(ds->dsSocket, buf, MAX_BUF_LEN-1 , 0,(struct sockaddr *) &originAddr, &addr_len)) == -1)
    {
        return -1;
    }
    
    if (numbytes==0)
    {
        return 0;
    }
    // Set the last element to \0 so it is a string
    buf[numbytes] = '\0';
    
#ifdef DEBUG
    printf("got packet from %s\n",inet_ntoa(originAddr.sin_addr));
    printf("interface: %s port: %d\n",ds->ifname,ntohs(originAddr.sin_port));
    printf("packet is %d bytes long\n",numbytes);
    printf("packet contains \"%s\"\n",buf);
#endif
    
    processDiscoveryServiceMsg(ds, buf, numbytes, originAddr);

    return 0;
}


/******************************************************************************
 * Purpose: This function checks for incomming messages for the discovery service
 *
 * Params: None
 *
 * Returns:  0 if OK
 *          -1 on Error
 *
 * Comments:
 *
 *****************************************************************************/
int checkDiscoveryService(unsigned int timeout)
{
    int res;
    struct timeval tv;
    fd_set readfd;

    tv.tv_sec = (timeout / 1000);
    tv.tv_usec = (timeout % 1000) * 1000;

    readfd = m_rfds;

    res = select(m_LastSocket + 1 , &readfd, NULL, NULL, &tv);
    if (res < 0 && errno != EINTR)
    {
        perror("select");
    }
    else if (res)
    {
        if (m_numPortConfig > 0 && FD_ISSET(m_dsPortMonitor[0].dsSocket, &readfd))
        {
            DSPortMonitor_Process(&m_dsPortMonitor[0]);
        }

        if (m_numPortConfig > 1 && FD_ISSET(m_dsPortMonitor[1].dsSocket, &readfd))
        {
            DSPortMonitor_Process(&m_dsPortMonitor[1]);
        }
    }
    
    return 0;
}

/******************************************************************************
 * Purpose: This function implements a thread function that executes
 *          the Discovery Service monitor function
 *
 * Params: None
 *
 * Returns:  0 if OK
 *          -1 on Error
 *
 * Comments:
 *
 *****************************************************************************/

void *threadFunction(void *arg)
{
    while (1)
    {
        checkDiscoveryService(PDS_POLLING_TIMEOUT);
    }
    return (NULL);
}


/******************************************************************************
 * Purpose: This function starts a thread that monitors the Discovery Service
 *
 * Params: None
 *
 * Returns:  0 if OK
 *          -1 on Error
 *
 * Comments:
 *
 *****************************************************************************/
int startDiscoveryServiceThread(void)
{
    if (pthread_create(&dsThread, NULL, threadFunction, NULL))
    {
        return -1;
    }
    
    return 0;
}


